<?php

namespace App\Http\Controllers;

use App\Enums\UserStates;
use App\Models\FailedLoginAttempt;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['login', 'register']]);
        $this->middleware('jwt', ['except' => ['login', 'register']]);
        $this->middleware('userStatus', ['except' => ['login', 'register']]);
        $this->middleware('blocked', ['except' => ['login', 'register']]);
    }


    public function register(Request $request): \Illuminate\Http\JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|between:2,50',
            'phone' => 'required|string|max:10|min:10|unique:users',
            'password' => 'required|string|confirmed|min:6',
            'address' => 'nullable|max:4000',
        ], [
            'name.required' => '11',
            'name.string' => '21',
            'name.between' => '22',
            'phone.required' => '11',
            'phone.string' => '21',
            'phone.max' => '23',
            'phone.min' => '24',
            'phone.unique' => '13',
            'password.required' => '11',
            'password.string' => '21',
            'password.confirmed' => '14',
            'password.min' => '25',
            'address.max' => '23',
            'notes.max' => '23'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors()->toJson(), 400);
        }

        $user = User::create(array_merge(
            $validator->validated(),
            ['password' => bcrypt($request->password)]
        ));

        return response()->json([
            'message' => 'User successfully registered',
            'user' => $user
        ], 200);

    }

    public function login(Request $request): \Illuminate\Http\JsonResponse
    {
        $credentials = $request->only('phone', 'password');

        if (!$token = auth()->attempt($credentials)) {
            FailedLoginAttempt::create([
                'user_id' => null, // Or get the user's ID
                'ip_address' => $request->ip(),
            ]);

            $consecutiveAttempts = FailedLoginAttempt::where('ip_address', $request->ip())
                ->where('created_at', '>=', now()->subMinutes(1))
                ->count();

            if ($consecutiveAttempts >= 3) {
                $user = User::where('phone', $request->phone)->first();

                $user->update([
                    'blocked_until' => now()->addMinutes(1),
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Too many failed login attempts. Your account has been blocked for 30 minutes.'
                ], 401);
            }

            return response()->json(['error' => 'Unauthorized'], 401);
        }

        return $this->respondWithToken($token);
    }

    public function me(): \Illuminate\Http\JsonResponse
    {
        auth()->user()->balance = OrdersController::getUserBalance(auth()->user()->id);
        return response()->json(auth()->user());
    }


    public function logout(): \Illuminate\Http\JsonResponse
    {
        auth()->logout();
        return response()->json(['message' => 'Successfully logged out']);
    }

    public function refresh(): \Illuminate\Http\JsonResponse
    {
        return $this->respondWithToken(auth()->refresh());
    }

    protected static function respondWithToken($token): \Illuminate\Http\JsonResponse
    {
        $user = User::find(auth()->id());
        if ($user->user_state == UserStates::$active) {
            return response()->json([
                'user_type' => $user->user_type,
                'access_token' => $token,
                'token_type' => 'bearer',
                'expires_in' => auth()->factory()->getTTL() * 60
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'your account is disable'
            ], 400);
        }
    }

}
