<?php

namespace App\Http\Controllers;

use App\Enums\OrderStates;
use App\Enums\UserTypes;
use App\Http\Requests\ChangeOrderState;
use App\Http\Requests\ChargeBalanceRequest;
use App\Http\Requests\FilterOrderRequest;
use App\Http\Requests\OrdersRequest;
use App\Models\Order;
use App\Models\Service;
use App\Models\User;
use Illuminate\Http\Client\Request;
use Illuminate\Support\Facades\DB;

class OrdersController extends Controller
{
    private string $path = 'images/orders/';

    public function __construct()
    {
        $this->middleware('jwt');
        $this->middleware('userStatus');
        $this->middleware('blocked');
    }

    public function getAll(): \Illuminate\Http\JsonResponse
    {
        $orders = Order::orderBy('order_status')->orderBy('id', 'desc')->take(50)->get();

        return response()->json([
            'success' => true,
            'data' => $orders,
        ], 200);
    }

    public function getUserOrders(): \Illuminate\Http\JsonResponse
    {
        $orders = Order::where('user_id', auth()->user()->id)->orderBy('id', 'desc')->take(100)->get();

        return response()->json([
            'success' => true,
            'data' => $orders,
        ], 200);
    }

    public function getOrdersByAdmin(): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_accept_auto_orders) {
            $orders = Order::where('order_status', OrderStates::$pending)->get();
            $autoPendingOrders = [];
            foreach ($orders as $order) {
                if ($order->service_code != null && $order->admin_id != null) {
                    if ($order->admin_id == auth()->user()->id) {
                        $order['service_code_with_data'] = str_replace(["N", "X", "P"], [$order['account_number'], $order['account_number_2'], $order['price']], $order['service_code']);
                        array_push($autoPendingOrders, $order);
                    }
                }
            }
            return response()->json([
                'success' => true,
                'data' => $autoPendingOrders,
            ], 200);

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }

    }

    public function add(OrdersRequest $request): \Illuminate\Http\JsonResponse
    {
        if ($request->has('image')) {
            $service = Service::find($request->service_id);

            if ($service != null) {

                if ($service->has_image) {

                    $imgName = time() . '.' . $request->image->getClientOriginalExtension();
                    $request->file('image')->move($this->path, $imgName);

                    $order = new Order(array_merge(
                        $request->validated(),
                        ['image' => $this->path . $imgName],
                    ));

                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'cant add image from this order for this service!'
                    ], 422);
                }

            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'service not exists!'
                ], 422);
            }

        } else {
            $order = new Order($request->validated());
        }

        $balance = self::getUserBalance($request->user_id);

        if ($request->total <= $balance) {
            if ($order->save()) {
                return response()->json([
                    'success' => true,
                    'data' => $order,
                ], 200);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'error saving order info',
                ], 422);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => 'your balance is smaller than order total',
            ], 400);
        }
    }

    public function changeState(ChangeOrderState $request, $id): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_edit_orders) {

            $order = Order::find($id);
            $balance = self::getUserBalance($order->user_id);

            if ($order != null && $order->order_status == OrderStates::$pending) {
                try {
                    DB::beginTransaction();
                   if ($order->service_id == null) {
                        if ($request->order_status == OrderStates::$completed) {
                            if (auth()->user()->balance_to_charge >= $order->total * -1) {
                                $user = User::find(auth()->user()->id);
                                $user->balance_to_charge = $user->balance_to_charge - ($order->total * -1);
                                $user->save();
                            } else {
                                return response()->json([
                                    'success' => false,
                                    'message' => 'can not charge for user because do not have enough balance!',
                                ], 422);
                            }
                        }
                    }


                    if ($balance < $order->total) {
                        $order->order_status = OrderStates::$rejected;
                        $order->rejection_note = "ليس لديك رصيد كافي!";
                    } else {
                        $order->order_status = $request->order_status;
                        $order->rejection_note = $request->rejection_note;
                    }

                    $order->admin_id_for_statistics = auth()->user()->id;

                    $order->save();

                    DB::commit();

                    return response()->json([
                        'success' => true,
                        'data' => $order
                    ], 200);

                } catch (\Exception $e) {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'error saving order status',
                    ], 422);
                }

            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'order not found!'
                ], 422);
            }

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }
    }

    public function chargeBalance(ChargeBalanceRequest $request): \Illuminate\Http\JsonResponse
    {
        try {
            DB::beginTransaction();

            if (auth()->user()->user_type == UserTypes::$admin) {

                if (auth()->user()->balance_to_charge >= $request->total) {
                    $user = User::find(auth()->user()->id);
                    $user->balance_to_charge = $user->balance_to_charge - $request->total;
                    $user->save();

                    $order = new Order(array_merge(
                        $request->validated(),
                        [
                            'total' => $request->total * -1,
                            'admin_id_for_statistics' => auth()->user()->id
                        ],
                    ));
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'can not charge for user because do not have enough balance!',
                    ], 422);
                }

            } else {

                $order = new Order(array_merge(
                    $request->validated(),
                    [
                        'total' => $request->total * -1,
                        'admin_id_for_statistics' => null
                    ],
                ));

            }

            $order->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'data' => $order
            ], 200);

        } catch (\Exception $e) {

            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'error saving order info',
            ], 422);
        }

    }

    public function filter(FilterOrderRequest $request): \Illuminate\Http\JsonResponse
    {
        if ($request->user_id != null && $request->service_id != null) {
            $orders = Order::where('user_id', $request->user_id)->where('service_id', $request->service_id)->whereBetween('created_at', [$request->start_date, $request->end_date])
                ->orderBy('order_status')->orderBy('id', 'desc')->get();
        } elseif ($request->user_id != null && $request->service_id == null) {
            $orders = Order::where('user_id', $request->user_id)->whereBetween('created_at', [$request->start_date, $request->end_date])
                ->orderBy('order_status')->orderBy('id', 'desc')->get();
        } elseif ($request->user_id == null && $request->service_id != null) {
            $orders = Order::where('service_id', $request->service_id)->whereBetween('created_at', [$request->start_date, $request->end_date])
                ->orderBy('order_status')->orderBy('id', 'desc')->get();
        } else {
            $orders = Order::whereBetween('created_at', [$request->start_date, $request->end_date])
                ->orderBy('order_status')->orderBy('id', 'desc')->get();
        }

        return response()->json([
            'success' => true,
            'data' => $orders
        ], 200);
    }

    public function checkIsTotalOrderAccepted($id): \Illuminate\Http\JsonResponse
    {
        $order = Order::find($id);

        if ($order != null) {

            $balance = self::getUserBalance($order->user_id);

            if ($order->total <= $balance) {
                return response()->json([
                    'success' => true,
                ], 200);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'user balance is not enough!',
                ], 422);
            }

        } else {
            return response()->json([
                'success' => false,
                'message' => 'order not found!',
            ], 422);
        }

    }

    public function getChargeBalanceByAdmin($id): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_see_statistics) {

            $orders = Order::where([['service_id', null], ['order_status', OrderStates::$completed], ['admin_id_for_statistics', $id]])->orderBy('id', 'desc')->get();

            $data = [
                'total' => $orders->sum('total') * -1,
                'count' => $orders->count(),
                'name' => $orders->first() != null ? $orders->first()->admin_name : '',
                'orders' => $orders,
            ];

            return response()->json([
                'success' => true,
                'data' => $data,
            ], 200);

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }

    }

    public function getByUser($id): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_see_statistics) {

            try {

                $orders = Order::where('user_id', $id)->orderBy('id', 'desc')->get();

                $data = [
                    'total' => $orders->sum('total'),
                    'count' => $orders->count(),
                    'name' => $orders->first()->user_name,
                    'orders' => $orders,
                ];

                return response()->json([
                    'success' => true,
                    'data' => $data,
                ], 200);

            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'orders not found!',
                ], 422);
            }

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }
    }

    public function getTotalBalance(): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_see_statistics) {

            $orders = Order::where('order_status', OrderStates::$completed);

            $data = [
                'total' => $orders->sum('total') * -1,
            ];

            return response()->json([
                'success' => true,
                'data' => $data,
            ], 200);

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }
    }

    public function getTotalBalanceByTowDates(Request $request): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_see_statistics) {

            $res = Order::where('order_status', OrderStates::$completed)->whereBetween('created_at', [$request->start_date, $request->end_date])->sum('total') * -1;

            $data = [
                'total' => $res,
            ];

            return response()->json([
                'success' => true,
                'data' => $data,
            ], 200);

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }
    }


    public function getByServiceOrUser(?int $serviceId, ?int $userId): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->user_type == UserTypes::$admin) {

            if ($serviceId == null && $userId == null) {
                return response()->json([
                    'success' => false,
                    'message' => 'user id and service id is null!'
                ], 422);
            }
            $orders = [];
            if ($serviceId == null && $userId != null) {
                $orders = Order::where('user_id', $userId)->orderBy('id', 'desc')->get();
            } elseif ($serviceId != null && $userId == null) {
                $orders = Order::where('service_id', $serviceId)->orderBy('id', 'desc')->get();
            } elseif ($serviceId != null && $userId != null) {
                $orders = Order::where([['service_id', $serviceId], ['user_id', $userId]])->orderBy('id', 'desc')->get();
            }

            return response()->json([
                'success' => true,
                'data' => $orders
            ], 200);

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you are not an admin!'
            ], 422);
        }
    }


    public static function getUserBalance($id)
    {
        return Order::where([['user_id', $id], ['order_status', OrderStates::$completed]])->sum('total') * -1;
    }

}
