<?php

namespace App\Http\Controllers;


use App\Enums\ServiceTypes;
use App\Enums\UserTypes;
use App\Http\Requests\ImagesRequest;
use App\Http\Requests\ServiceEditRequest;
use App\Http\Requests\ServicesRequest;
use App\Models\Order;
use App\Models\Service;

class ServicesController extends Controller
{
    private string $path = 'images/services/';

    public function __construct()
    {
        $this->middleware('jwt');
        $this->middleware('userStatus');
        $this->middleware('blocked');
    }

    public function getAll(): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->user_type == UserTypes::$admin) {
            $services = Service::where('type', ServiceTypes::$main)->get();
        } else {
            $services = Service::where([['type', ServiceTypes::$main], ['is_active', true]])->get();
        }

        foreach ($services as $service) {
            if ($service->type == ServiceTypes::$main) {
                if (auth()->user()->user_type == UserTypes::$admin) {
                    $childs = Service::where('parent_id', $service->id)->get();
                    foreach ($childs as $child) {
                        $child->parent_name = $service->name;
                    }
                    $service->childs = $childs;
                } else {
                    $service->childs = Service::where([['parent_id', $service->id], ['is_active', true]])->get();
                }
            }
        }

        return response()->json([
            'success' => true,
            'data' => $services,
        ], 200);
    }

    public function getAllByCategory($id): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->user_type == UserTypes::$admin) {
            $services = Service::where([['type', ServiceTypes::$main], ['category_id', $id]])->get();
        } else {
            $services = Service::where([['type', ServiceTypes::$main], ['is_active', true], ['category_id', $id]])->get();
        }
        foreach ($services as $service) {
            if ($service->type == ServiceTypes::$main) {
                if (auth()->user()->user_type == UserTypes::$admin) {
                    $service->childs = Service::where('parent_id', $service->id)->get();
                } else {
                    $service->childs = Service::where([['parent_id', $service->id], ['is_active', true]])->get();
                }
            }
        }
        return response()->json([
            'success' => true,
            'data' => $services,
        ], 200);
    }

    public function getAllParent(): \Illuminate\Http\JsonResponse
    {
        $parentServices = service::where('type', ServiceTypes::$main)->select('id', 'name')->get();
        if ($parentServices->count() == 0) {
            return response()->json([
                'success' => false,
            ], 200);
        } else {
            return response()->json([
                'success' => true,
                'data' => $parentServices,
            ], 200);
        }
    }


    public function add(ServicesRequest $request): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_add_service) {
            $service = new Service($request->validated());
            if ($service->save()) {
                return response()->json([
                    'success' => true,
                    'data' => $service
                ], 200);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'error saving service info',
                ], 422);
            }
        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!',
            ], 422);
        }
    }

    public function edit(ServiceEditRequest $request, $id): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_edit_service) {
            $service = Service::find($id);

            if ($service != null) {

                $service->name = $request->name;
                $service->description = $request->description;
                $service->notes = $request->notes;
                $service->cost = $request->cost;
                $service->tax_percent = $request->tax_percent;
                $service->service_code = $request->service_code;
                $service->sim_id = $request->sim_id;
                $service->admin_id = $request->admin_id;
                $service->category_id = $request->category_id;
                $service->is_active = $request->is_active;
                $service->has_second_account_number_2 = $request->has_second_account_number_2;

                if ($service->save()) {
                    return response()->json([
                        'success' => true,
                        'data' => $service
                    ], 200);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'error saving service info',
                    ], 422);
                }

            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'service not found!'
                ], 422);
            }

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!',
            ], 422);
        }
    }

    public function setImage(ImagesRequest $request, $id): \Illuminate\Http\JsonResponse
    {
        $service = Service::find($id);

        if (isset($service->img_path)) {
            unlink($service->img_path);
        }

        $imgName = time() . '.' . $request->img_path->getClientOriginalExtension();
        $request->file('img_path')->move($this->path, $imgName);
        $service['img_path'] = $this->path . $imgName;

        if ($service->save()) {
            return response()->json([
                'success' => true,
                'data' => $service
            ], 200);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'error saving service image',
            ], 422);
        }

    }

    public function delete($id): \Illuminate\Http\JsonResponse
    {
        if (auth()->user()->can_delete_service) {
            $service = Service::find($id);

            if ($service != null) {

                $hasOrders = Order::where('service_id', $id)->exists();
                $hasChilds = Service::where('parent_id', $id)->exists();

                if (!$hasOrders && !$hasChilds) {

                    if ($service->delete()) {

                        if (isset($service->img_path)) {
                            unlink($service->img_path);
                        }

                        return response()->json([
                            'success' => true,
                        ], 200);

                    } else {

                        return response()->json([
                            'success' => false,
                            'message' => 'error deleting service info',
                        ], 422);

                    }
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => 'can not delete service!'
                    ], 422);
                }
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'service not found!'
                ], 422);
            }

        } else {
            return response()->json([
                'success' => false,
                'message' => 'you do not have permission to do this!'
            ], 422);
        }
    }

}
